import Fastify from "fastify";
import fastifyWebsocket from "@fastify/websocket";
import fastifyStatic from "@fastify/static";
import pino from "pino";
import { createCaptureClient } from "./grpc/client.js";
import { registerHttpRoutes } from "./http/routes.js";
import { registerWsRoute } from "./ws/server.js";
import path from "node:path";
import { fileURLToPath } from "node:url";
const __dirname = path.dirname(fileURLToPath(import.meta.url));
const PORT = Number(process.env.BRIDGE_PORT ?? 5001);
const AGENT_ADDRESS = process.env.AGENT_ADDRESS ?? "127.0.0.1";
const AGENT_PORT = Number(process.env.AGENT_PORT ?? 5002);
const CACHE_DIR = path.resolve(process.cwd(), ".cache");
const STATIC_UI_DIR = process.env.STATIC_UI_DIR;
const logger = pino({
    level: process.env.LOG_LEVEL ?? "info",
    transport: process.env.NODE_ENV !== "production"
        ? { target: "pino-pretty", options: { colorize: true } }
        : undefined,
});
let client = null;
async function connectAgent() {
    try {
        client = createCaptureClient({ address: AGENT_ADDRESS, port: AGENT_PORT });
        logger.info({ address: `${AGENT_ADDRESS}:${AGENT_PORT}` }, "gRPC client created");
    }
    catch (e) {
        logger.warn({ err: e }, "Agent not available");
        client = null;
    }
}
async function main() {
    await connectAgent();
    const app = Fastify({ logger: false });
    await app.register(fastifyWebsocket);
    registerHttpRoutes(app, client, logger);
    registerWsRoute(app, () => client, logger, CACHE_DIR);
    if (STATIC_UI_DIR) {
        const uiRoot = path.resolve(STATIC_UI_DIR);
        await app.register(fastifyStatic, { root: uiRoot });
        app.get("/", (_req, reply) => reply.sendFile("index.html"));
        logger.info({ uiRoot }, "Serving static UI");
    }
    await app.listen({ host: "127.0.0.1", port: PORT });
    logger.info({ port: PORT, cacheDir: CACHE_DIR }, "Bridge listening");
}
main().catch((err) => {
    logger.error(err);
    process.exit(1);
});
//# sourceMappingURL=index.js.map