import { createSession } from "../sessions/create.js";
import { getSession } from "../db/sessions.js";
import { transitionSession } from "../sessions/stateMachine.js";
import { storeCaptureResult } from "../captures/storeCapture.js";
import { validateSession } from "../rules/validate.js";
import { submit } from "../vendor/submit.js";
import { getSupabase } from "../db/client.js";
export async function registerSessionRoutes(app) {
    app.post("/sessions", async (req, reply) => {
        try {
            const { ori, purpose, operator_id, operator_name, device_id, applicant_id } = req.body ?? {};
            if (!ori || !purpose)
                return reply.status(400).send({ error: "ori and purpose required" });
            const result = await createSession({ ori, purpose, operator_id, operator_name, device_id, applicant_id });
            return reply.send(result);
        }
        catch (e) {
            req.log.error(e, "create session");
            return reply.status(500).send({ error: e instanceof Error ? e.message : "Internal error" });
        }
    });
    app.get("/sessions/:id", async (req, reply) => {
        try {
            const session = await getSession(req.params.id);
            if (!session)
                return reply.status(404).send({ error: "Session not found" });
            return reply.send(session);
        }
        catch (e) {
            req.log.error(e, "get session");
            return reply.status(500).send({ error: e instanceof Error ? e.message : "Internal error" });
        }
    });
    app.patch("/sessions/:id/status", async (req, reply) => {
        try {
            const result = await transitionSession(req.params.id, req.body.status, undefined, req.body.tcn);
            if (!result.ok)
                return reply.status(400).send({ error: result.error });
            return reply.send({ ok: true });
        }
        catch (e) {
            req.log.error(e, "transition status");
            return reply.status(500).send({ error: e instanceof Error ? e.message : "Internal error" });
        }
    });
    app.post("/sessions/:id/demographics", async (req, reply) => {
        try {
            const sessionId = req.params.id;
            const session = await getSession(sessionId);
            if (!session)
                return reply.status(404).send({ error: "Session not found" });
            const { error } = await getSupabase().from("subject_demographics").upsert({
                session_id: sessionId,
                ...req.body,
                updated_at: new Date().toISOString(),
            }, { onConflict: "session_id" });
            if (error)
                return reply.status(400).send({ error: error.message });
            await transitionSession(sessionId, "DEMOGRAPHICS_ENTERED");
            return reply.send({ ok: true });
        }
        catch (e) {
            req.log.error(e, "demographics");
            return reply.status(500).send({ error: e instanceof Error ? e.message : "Internal error" });
        }
    });
    app.post("/sessions/:id/captures", async (req, reply) => {
        try {
            const sessionId = req.params.id;
            const body = req.body;
            const wsqBuffer = Buffer.from(body.wsq_base64, "base64");
            await storeCaptureResult({
                session_id: sessionId,
                finger_position: body.finger_position,
                capture_kind: body.capture_kind,
                attempt: body.attempt,
                quality_score: body.quality_score,
                quality_type: body.quality_type,
                wsq_buffer: wsqBuffer,
                width: body.width,
                height: body.height,
                captured_at: body.captured_at,
            });
            return reply.send({ ok: true });
        }
        catch (e) {
            req.log.error(e, "store capture");
            return reply.status(500).send({ error: e instanceof Error ? e.message : "Internal error" });
        }
    });
    app.get("/sessions/:id/validate", async (req, reply) => {
        try {
            const session = await getSession(req.params.id);
            if (!session)
                return reply.status(404).send({ error: "Session not found" });
            const result = await validateSession(req.params.id, session.ori);
            return reply.send(result);
        }
        catch (e) {
            req.log.error(e, "validate");
            return reply.status(500).send({ error: e instanceof Error ? e.message : "Internal error" });
        }
    });
    app.post("/sessions/:id/submit", async (req, reply) => {
        try {
            const vendor = req.body?.vendor ?? "stub";
            const result = await submit(req.params.id, vendor);
            return reply.send(result);
        }
        catch (e) {
            req.log.error(e, "submit");
            return reply.status(500).send({ error: e instanceof Error ? e.message : "Internal error" });
        }
    });
}
//# sourceMappingURL=routes.js.map