import { createCipheriv, createDecipheriv, randomBytes, scryptSync } from "node:crypto";
import fs from "node:fs/promises";
import path from "node:path";
const ALG = "aes-256-gcm";
const KEY_LEN = 32;
const IV_LEN = 16;
const SALT_LEN = 32;
function getKey() {
    const raw = process.env.ARTIFACT_KEY;
    if (raw && raw.length >= 64 && /^[0-9a-fA-F]+$/.test(raw)) {
        return Buffer.from(raw.slice(0, 64), "hex");
    }
    const fallback = process.env.ARTIFACT_KEY_FALLBACK ?? "ls360-artifact-fallback-change-me";
    return scryptSync(fallback, "ls360-salt", KEY_LEN);
}
export function getArtifactsDir() {
    const base = process.env.ARTIFACTS_DIR ?? path.join(process.cwd(), ".artifacts");
    return path.join(base, "encrypted");
}
export async function writeEncrypted(sessionId, fingerPosition, attempt, data, expiresAt) {
    const dir = getArtifactsDir();
    const ref = `${sessionId}/${fingerPosition}_${attempt}.enc`;
    const fullPath = path.join(dir, ref);
    await fs.mkdir(path.dirname(fullPath), { recursive: true });
    const key = getKey();
    const iv = randomBytes(IV_LEN);
    const cipher = createCipheriv(ALG, key, iv);
    const encrypted = Buffer.concat([cipher.update(data), cipher.final(), cipher.getAuthTag()]);
    const meta = JSON.stringify({ iv: iv.toString("base64"), expiresAt: expiresAt.toISOString() });
    await fs.writeFile(fullPath, Buffer.concat([Buffer.from(meta, "utf8"), Buffer.from("\n"), encrypted]));
    return ref;
}
export async function readDecrypted(ref) {
    const fullPath = path.join(getArtifactsDir(), ref);
    const raw = await fs.readFile(fullPath);
    const idx = raw.indexOf("\n");
    if (idx <= 0)
        throw new Error("Invalid artifact file");
    const meta = JSON.parse(raw.subarray(0, idx).toString("utf8"));
    const enc = raw.subarray(idx + 1);
    const key = getKey();
    const iv = Buffer.from(meta.iv, "base64");
    const tag = enc.subarray(enc.length - 16);
    const body = enc.subarray(0, enc.length - 16);
    const decipher = createDecipheriv(ALG, key, iv);
    decipher.setAuthTag(tag);
    return Buffer.concat([decipher.update(body), decipher.final()]);
}
export async function deleteArtifact(ref) {
    const fullPath = path.join(getArtifactsDir(), ref);
    await fs.unlink(fullPath).catch(() => { });
}
//# sourceMappingURL=encrypt.js.map