#Requires -Version 5.1
<#
.SYNOPSIS
    Extracts the RealScan SDK zip into agent/libs/RealScanSdk for USB integration.
.DESCRIPTION
    Run this script with the path to "RealScan SDK for Windows(US) v2.2.0.2469.zip"
    (or your downloaded zip). It will extract into the project's agent/libs/RealScanSdk folder.
.EXAMPLE
    .\Extract-RealScanSdk.ps1 -ZipPath "C:\Users\You\Downloads\RealScan SDK for Windows(US) v2.2.0.2469.zip"
.EXAMPLE
    .\Extract-RealScanSdk.ps1
    (Prompts for zip path or searches Downloads and Desktop)
#>
param(
    [Parameter(Mandatory = $false)]
    [string] $ZipPath
)

$ErrorActionPreference = "Stop"
$scriptDir = $PSScriptRoot
$repoRoot = if (Test-Path (Join-Path $scriptDir "..\..\..\agent")) { (Resolve-Path (Join-Path $scriptDir "..\..\..")).Path } else { $scriptDir }
$targetDir = Join-Path $repoRoot "agent\libs\RealScanSdk"

if (-not $ZipPath) {
    $candidates = @(
        (Join-Path $env:USERPROFILE "Downloads\RealScan SDK for Windows(US) v2.2.0.2469.zip"),
        (Join-Path $env:USERPROFILE "Downloads\RealScan*.zip"),
        (Join-Path $env:USERPROFILE "OneDrive\Desktop\RealScan*.zip")
    )
    foreach ($c in $candidates) {
        $found = Get-Item $c -ErrorAction SilentlyContinue
        if ($found) {
            $ZipPath = $found | Select-Object -First 1 -ExpandProperty FullName
            Write-Host "Using zip: $ZipPath"
            break
        }
    }
    if (-not $ZipPath) {
        Write-Host "RealScan SDK zip not found. Please run with -ZipPath 'C:\path\to\RealScan SDK for Windows(US) v2.2.0.2469.zip'"
        exit 1
    }
}

if (-not (Test-Path $ZipPath)) {
    Write-Error "Zip not found: $ZipPath"
    exit 1
}

if (-not (Test-Path $targetDir)) {
    New-Item -ItemType Directory -Path $targetDir -Force | Out-Null
}

Write-Host "Extracting to: $targetDir"
Expand-Archive -Path $ZipPath -DestinationPath $targetDir -Force

# If zip had a single root folder (e.g. "RealScan SDK 2.2"), flatten so Bin is at targetDir\Bin
$children = Get-ChildItem -Path $targetDir -Directory
if ($children.Count -eq 1 -and (Get-ChildItem -Path $targetDir -File).Count -eq 0) {
    $inner = $children[0].FullName
    Get-ChildItem -Path $inner | Move-Item -Destination $targetDir -Force
    Remove-Item -Path $inner -Force -Recurse -ErrorAction SilentlyContinue
    Write-Host "Flattened single root folder."
}

Write-Host "Contents:"
Get-ChildItem -Path $targetDir -Recurse | Select-Object FullName | ForEach-Object { $_.FullName }
Write-Host "Done. Set UseGSdk: false in appsettings.json, restart the Agent, and use Refresh in the app to see USB devices."

